# Deployment Guide - Aerial Vision

## Quick Deployment Steps

### 1. Install Dependencies

**Frontend:**
```bash
npm install
```

**Backend:**
```bash
cd server
npm install
cd ..
```

### 2. Build Frontend

```bash
npm run build
```

This creates a `dist/` folder with all static files.

### 3. Configure Environment

Create `.env` file in root (optional, defaults work for single-server setup):

```env
VITE_API_URL=/api
PORT=3001
```

### 4. Start Server

The server serves both API and frontend:

```bash
cd server
npm start
```

Or for development:
```bash
npm run dev
```

### 5. Access Application

- Frontend: `http://localhost:3001`
- API: `http://localhost:3001/api`

## Production Deployment

### Option A: Single Server (Recommended)

1. **Upload all files** to your server
2. **Install dependencies**:
   ```bash
   npm install
   cd server && npm install && cd ..
   ```
3. **Build frontend**:
   ```bash
   npm run build
   ```
4. **Start server**:
   ```bash
   cd server
   npm start
   ```
5. **Use PM2** (recommended for production):
   ```bash
   npm install -g pm2
   cd server
   pm2 start server.js --name aerial-vision
   pm2 save
   pm2 startup
   ```

### Option B: Reverse Proxy (Nginx)

**Nginx Configuration:**

```nginx
server {
    listen 80;
    server_name yourdomain.com;

    # Increase upload size limit
    client_max_body_size 500M;

    # Serve static files (frontend)
    location / {
        root /path/to/nira-lone/dist;
        try_files $uri $uri/ /index.html;
    }

    # Proxy API requests
    location /api {
        proxy_pass http://localhost:3001;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;
        
        # Increase timeouts for large file uploads
        proxy_read_timeout 300s;
        proxy_connect_timeout 300s;
    }

    # Serve model files with CORS
    location /api/models/ {
        proxy_pass http://localhost:3001;
        proxy_set_header Host $host;
        add_header Access-Control-Allow-Origin *;
        add_header Access-Control-Allow-Methods 'GET, OPTIONS';
    }
}
```

**Then:**
1. Build frontend: `npm run build`
2. Copy `dist/` to `/path/to/nira-lone/dist`
3. Start backend: `cd server && npm start` (or use PM2)
4. Reload Nginx: `sudo nginx -s reload`

### Option C: Separate Frontend/Backend

**Backend Deployment:**
1. Upload `server/` folder
2. Install: `cd server && npm install`
3. Set `PORT` environment variable
4. Start: `npm start` or use PM2

**Frontend Deployment:**
1. Build: `npm run build`
2. Set `VITE_API_URL` to your backend URL (e.g., `https://api.yourdomain.com/api`)
3. Deploy `dist/` folder to:
   - Vercel: `vercel --prod`
   - Netlify: Drag `dist` folder
   - Any static host

## File Permissions

Ensure uploads directory is writable:

```bash
chmod 755 server/uploads
chmod 755 server/uploads/models
```

## Database

SQLite database is created automatically at `server/database.sqlite`.

For production, consider:
- Regular backups
- Moving to PostgreSQL/MySQL for better performance
- Using environment variables for database path

## Security Considerations

1. **File Upload Limits**: Currently 500MB max (adjust in `server.js`)
2. **CORS**: Configure allowed origins in production
3. **Rate Limiting**: Add rate limiting middleware
4. **Authentication**: Add user authentication for production
5. **HTTPS**: Use SSL/TLS certificates (Let's Encrypt)

## Troubleshooting

**Port Already in Use:**
```bash
# Find process using port 3001
lsof -i :3001
# Kill it or change PORT in .env
```

**Upload Fails:**
- Check file size limits (Nginx: `client_max_body_size`)
- Check disk space: `df -h`
- Check permissions: `ls -la server/uploads`

**Models Not Loading:**
- Check CORS headers
- Verify API URL in browser console
- Check server logs

## Performance Tips

1. **Enable Compression**: Already enabled in server.js
2. **CDN**: Use CDN for static assets
3. **Caching**: Add cache headers for model files
4. **Database**: Consider PostgreSQL for large datasets
5. **File Storage**: Consider S3/Cloudflare R2 for large files

## Monitoring

Use PM2 monitoring:
```bash
pm2 monit
pm2 logs aerial-vision
```

Or add logging to server.js for production monitoring.

